/************************* MPEG-2 NBC Audio Decoder **************************
 *                                                                           *
"This software module was originally developed by
AT&T, Dolby Laboratories, Fraunhofer Gesellschaft IIS in the course of
development of the MPEG-2 NBC/MPEG-4 Audio standard ISO/IEC 13818-7,
14496-1,2 and 3. This software module is an implementation of a part of one or more
MPEG-2 NBC/MPEG-4 Audio tools as specified by the MPEG-2 NBC/MPEG-4
Audio standard. ISO/IEC  gives users of the MPEG-2 NBC/MPEG-4 Audio
standards free license to this software module or modifications thereof for use in
hardware or software products claiming conformance to the MPEG-2 NBC/MPEG-4
Audio  standards. Those intending to use this software module in hardware or
software products are advised that this use may infringe existing patents.
The original developer of this software module and his/her company, the subsequent
editors and their companies, and ISO/IEC have no liability for use of this software
module or modifications thereof in an implementation. Copyright is not released for
non MPEG-2 NBC/MPEG-4 Audio conforming products.The original developer
retains full right to use the code for his/her  own purpose, assign or donate the
code to a third party and to inhibit third party from using the code for non
MPEG-2 NBC/MPEG-4 Audio conforming products. This copyright notice must
be included in all copies or derivative works."
Copyright(c)1996.
 *                                                                           *
 ****************************************************************************/
/*
 * $Id: filtbank.c,v 1.12 2006/12/05 07:06:47 kyang Exp $
 */

/*
 * CHANGES:
 *  2001/01/17: menno: Added frequency cut off filter.
 *
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "fixed.h"
#include "coder.h"
#include "filtbank.h"
#include "frame.h"
#include "fft.h"
#include "util.h"

static const frac_t sine_long_1024[1024];

#ifdef FAAC_STATIC_MEMORY
static coef_t faac_freqBuff[2][FRAME_LEN<<1];
static pow_t faac_freqBuff2[2][FRAME_LEN<<1];
static coef_t xi[512];
static coef_t xr[512];
#else
static coef_t *xi;
static coef_t *xr;
#endif

void FilterBankInit(faacEncHandle hEncoder)
{
    uint32_t channel;
    for (channel = 0; channel < hEncoder->numChannels; channel++) {
#ifdef FAAC_STATIC_MEMORY
        hEncoder->freqBuff[channel] = (coef_t*)faac_freqBuff[channel];
        hEncoder->freqBuff2[channel] = (pow_t*)faac_freqBuff2[channel];
#else
        hEncoder->freqBuff[channel] = (coef_t*)AllocMemory((FRAME_LEN<<1)*sizeof(coef_t));
        hEncoder->freqBuff2[channel] = (pow_t*)AllocMemory((FRAME_LEN<<1)*sizeof(pow_t));
#endif
    }
#ifndef FAAC_STATIC_MEMORY
    xi = (coef_t *)AllocMemory(512*sizeof(coef_t));
    xr = (coef_t *)AllocMemory(512*sizeof(coef_t));
#endif
}

void FilterBankEnd(faacEncHandle hEncoder)
{
#ifndef FAAC_STATIC_MEMORY
    uint32_t channel;
    for (channel = 0; channel < hEncoder->numChannels; channel++) {
        if ( hEncoder->freqBuff[channel] )
            FreeMemory(hEncoder->freqBuff[channel]);
        if ( hEncoder->freqBuff2[channel] )
            FreeMemory(hEncoder->freqBuff2[channel]);
    }
    if ( xi )
        FreeMemory(xi);
    if ( xr )
        FreeMemory(xr);
#endif
}

static void MDCT( FFT_Tables *fft_tables, coef_t *data, int32_t N )
{
    static real_32_t *cosx = faac_fft_cosx;
    static real_32_t *sinx = faac_fft_sinx;
    coef_t tempr, tempi; /* temps for pre and post twiddle */
    register int32_t i;
    int32_t n,m;
    int32_t N2 = (N>>1);
    int32_t N4 = (N>>2);
    int32_t N8 = (N>>3);
#ifdef FAAC_USE_ASM
   int32_t hi,lo;
#endif

    for (i = 0; i < N4; i++) {
        /* calculate real and imaginary parts of g(n) or G(p) */
        n = N2 - 1 - (i<<1);
        m = (i<<1);

        if (i < N8) {
            tempr = (data [N4 + n] + data [N + N4 - 1 - n]);
            tempi = (data [N4 + m] - data [N4 - 1 - m]);
        }
        else {
            tempr = (data [N4 + n] - data [N4 - 1 - n]);
            tempi = (data [N4 + m] + data [N + N4 - 1 - m]);
        }

        tempr = tempr >> 1;
        tempi = tempi >> 1;

        /* calculate pre-twiddled FFT input */
#ifdef FAAC_USE_ASM
        FIXED_MUL(hi,lo,tempr,cosx[i]);
        FIXED_MADD(hi,lo,tempi,sinx[i]);
        xr[i] = FIXED_INT64_R(hi,lo);
        
        FIXED_MUL(hi,lo,tempi,cosx[i]);
        FIXED_MSUB(hi,lo,tempr,sinx[i]);
        xi[i] = FIXED_INT64_R(hi,lo);
#else
        xr[i] = MUL_R(tempr,cosx[i]) + MUL_R(tempi,sinx[i]);
        xi[i] = MUL_R(tempi,cosx[i]) - MUL_R(tempr,sinx[i]);
#endif
#ifdef DUMP_FFT
        printf("i = %d, xr = %.8f, xi = %.8f, tempr = %.8f, tempi = %.8f, cosx = %.8f, sinx = %.8f\n",
                i, COEF2FLOAT(xr[i]), COEF2FLOAT(xi[i]), 
                COEF2FLOAT(tempr), COEF2FLOAT(tempi), 
                REAL2FLOAT(cosx[i]), REAL2FLOAT(sinx[i]));
#endif
    }

    /* Perform in-place complex FFT of length N/4 */
    fft( fft_tables, xr, xi, FAAC_FFT_LOGM);

    /* post-twiddle FFT output and then get output data */
    for (i = 0; i < N4; i++) {
        /* fill in output values */
        data [N - 1 - (i<<1)] = (MUL_R(xr[i],cosx[i]) + MUL_R(xi[i],sinx[i]))<<2;  /* second half odd */
        data [i<<1] = -data [N - 1 - (i<<1)];   /* first half even */
        data [N2 - 1 - (i<<1)] = (MUL_R(xi[i],cosx[i]) - MUL_R(xr[i],sinx[i]))<<2;  /* first half odd */
        data [N2 + (i<<1)] = -data [N2 - 1 - (i<<1)];  /* second half even */
#ifdef DUMP_FFT
        printf("i = %d, data1 = %.8f, data2 = %.8f, xr = %.8f, xi = %.8f, cosx = %.8f, sinx = %.8f\n",
                i, COEF2FLOAT(data [N - 1 - (i<<1)]), COEF2FLOAT(data [N2 - 1 - (i<<1)]), 
                COEF2FLOAT(xr[i]), COEF2FLOAT(xi[i]), 
                REAL2FLOAT(cosx[i]), REAL2FLOAT(sinx[i]));
#endif
    }
#ifdef DUMP_FFT
    exit(1);
#endif
}

void FilterBank(faacEncHandle hEncoder,
                CoderInfo *coderInfo,
                int16_t *p_in_data,
                coef_t *p_out_mdct,
                int16_t *p_overlap,
                int32_t overlap_select)
{
    register int32_t i;
#ifdef FAAC_USE_ASM
    register int32_t hi,lo;
    register int32_t data;
#endif

    /* Separate action for each Block Type */
    for ( i = 0 ; i < BLOCK_LEN_LONG ; i++){
#ifdef FAAC_USE_ASM
        data = p_overlap[i<<1];
        FIXED_MUL(hi,lo,data,sine_long_1024[i]);
        p_out_mdct[i] = ((hi<<9)|((lo>>23)&0x1ff));
#else
        p_out_mdct[i] = (p_overlap[i<<1]*((int64_t)sine_long_1024[i]))>>FRAC2COEF_BIT;
#endif

#ifdef DUMP_P_O_MDCT
        printf("i=%d:  p_o_buf = %.8f, \tp_out_mdct = %.8f,\tfirst_window[i] = %.8f\n",
                    i, (double)p_overlap[i<<1], COEF2FLOAT(p_out_mdct[i]),FRAC2FLOAT(sine_long_1024[i]));
#endif
#ifdef FAAC_USE_ASM
        data = p_in_data[(BLOCK_LEN_LONG-i-1)<<1];
        FIXED_MUL(hi,lo,data,sine_long_1024[i]);
        p_out_mdct[(BLOCK_LEN_LONG<<1)-i-1] = ((hi<<9)|((lo>>23)&0x1ff));
#else
        p_out_mdct[(BLOCK_LEN_LONG<<1)-i-1] = (p_in_data[(BLOCK_LEN_LONG-i-1)<<1]*((int64_t)sine_long_1024[i]))>>FRAC2COEF_BIT;
#endif
#ifdef DUMP_P_O_MDCT
        printf("i=%d:+ p_o_buf = %.8f, \tp_out_mdct = %.8f,\tsecond_window = %.8f\n",
                    i, (double)p_in_data[(BLOCK_LEN_LONG-i-1)<<1], COEF2FLOAT(p_out_mdct[(BLOCK_LEN_LONG<<1)-i-1]),FRAC2FLOAT(sine_long_1024[i]));
#endif
    }
    MDCT( &hEncoder->fft_tables, p_out_mdct, BLOCK_LEN_LONG<<1 );
#ifdef DUMP_P_O_MDCT
//    exit(1);
#endif
}

void specFilter(coef_t *freqBuff,
                int32_t sampleRate,
                int32_t lowpassFreq,
                int32_t specLen
                )
{
    int32_t lowpass,xlowpass;

    /* calculate the last line which is not zero */
    lowpass = (lowpassFreq * specLen) / (sampleRate>>1) + 1;
    xlowpass = (lowpass < specLen) ? lowpass : specLen ;

    SetMemory(freqBuff+xlowpass,0,(specLen-xlowpass)*sizeof(coef_t));
}

void FilterBankPow(int32_t N, coef_t* data, pow_t* data2)
{
    register int32_t i;

    for (i = 0; i < N; i++) {
        data2[i] = (int64_t)(COEF2INT(data[i]))*COEF2INT(data[i]);
    }
}

ALIGN static const frac_t sine_long_1024[1024] =
{
    FRAC_CONST(0.00076699031874270449),
    FRAC_CONST(0.002300969151425805),
    FRAC_CONST(0.0038349425697062275),
    FRAC_CONST(0.0053689069639963425),
    FRAC_CONST(0.0069028587247297558),
    FRAC_CONST(0.0084367942423697988),
    FRAC_CONST(0.0099707099074180308),
    FRAC_CONST(0.011504602110422714),
    FRAC_CONST(0.013038467241987334),
    FRAC_CONST(0.014572301692779064),
    FRAC_CONST(0.016106101853537287),
    FRAC_CONST(0.017639864115082053),
    FRAC_CONST(0.019173584868322623),
    FRAC_CONST(0.020707260504265895),
    FRAC_CONST(0.022240887414024961),
    FRAC_CONST(0.023774461988827555),
    FRAC_CONST(0.025307980620024571),
    FRAC_CONST(0.026841439699098531),
    FRAC_CONST(0.028374835617672099),
    FRAC_CONST(0.029908164767516555),
    FRAC_CONST(0.031441423540560301),
    FRAC_CONST(0.032974608328897335),
    FRAC_CONST(0.03450771552479575),
    FRAC_CONST(0.036040741520706229),
    FRAC_CONST(0.037573682709270494),
    FRAC_CONST(0.039106535483329888),
    FRAC_CONST(0.040639296235933736),
    FRAC_CONST(0.042171961360347947),
    FRAC_CONST(0.043704527250063421),
    FRAC_CONST(0.04523699029880459),
    FRAC_CONST(0.046769346900537863),
    FRAC_CONST(0.048301593449480144),
    FRAC_CONST(0.049833726340107277),
    FRAC_CONST(0.051365741967162593),
    FRAC_CONST(0.052897636725665324),
    FRAC_CONST(0.054429407010919133),
    FRAC_CONST(0.055961049218520569),
    FRAC_CONST(0.057492559744367566),
    FRAC_CONST(0.059023934984667931),
    FRAC_CONST(0.060555171335947788),
    FRAC_CONST(0.062086265195060088),
    FRAC_CONST(0.063617212959193106),
    FRAC_CONST(0.065148011025878833),
    FRAC_CONST(0.066678655793001557),
    FRAC_CONST(0.068209143658806329),
    FRAC_CONST(0.069739471021907307),
    FRAC_CONST(0.071269634281296401),
    FRAC_CONST(0.072799629836351673),
    FRAC_CONST(0.074329454086845756),
    FRAC_CONST(0.075859103432954447),
    FRAC_CONST(0.077388574275265049),
    FRAC_CONST(0.078917863014784942),
    FRAC_CONST(0.080446966052950014),
    FRAC_CONST(0.081975879791633066),
    FRAC_CONST(0.083504600633152432),
    FRAC_CONST(0.085033124980280275),
    FRAC_CONST(0.08656144923625117),
    FRAC_CONST(0.088089569804770507),
    FRAC_CONST(0.089617483090022959),
    FRAC_CONST(0.091145185496681005),
    FRAC_CONST(0.09267267342991331),
    FRAC_CONST(0.094199943295393204),
    FRAC_CONST(0.095726991499307162),
    FRAC_CONST(0.097253814448363271),
    FRAC_CONST(0.098780408549799623),
    FRAC_CONST(0.10030677021139286),
    FRAC_CONST(0.10183289584146653),
    FRAC_CONST(0.10335878184889961),
    FRAC_CONST(0.10488442464313497),
    FRAC_CONST(0.10640982063418768),
    FRAC_CONST(0.10793496623265365),
    FRAC_CONST(0.10945985784971798),
    FRAC_CONST(0.11098449189716339),
    FRAC_CONST(0.11250886478737869),
    FRAC_CONST(0.1140329729333672),
    FRAC_CONST(0.11555681274875526),
    FRAC_CONST(0.11708038064780059),
    FRAC_CONST(0.11860367304540072),
    FRAC_CONST(0.1201266863571015),
    FRAC_CONST(0.12164941699910553),
    FRAC_CONST(0.12317186138828048),
    FRAC_CONST(0.12469401594216764),
    FRAC_CONST(0.12621587707899035),
    FRAC_CONST(0.12773744121766231),
    FRAC_CONST(0.12925870477779614),
    FRAC_CONST(0.13077966417971171),
    FRAC_CONST(0.13230031584444465),
    FRAC_CONST(0.13382065619375472),
    FRAC_CONST(0.13534068165013421),
    FRAC_CONST(0.13686038863681638),
    FRAC_CONST(0.13837977357778389),
    FRAC_CONST(0.13989883289777721),
    FRAC_CONST(0.14141756302230302),
    FRAC_CONST(0.14293596037764267),
    FRAC_CONST(0.14445402139086047),
    FRAC_CONST(0.14597174248981221),
    FRAC_CONST(0.14748912010315357),
    FRAC_CONST(0.14900615066034845),
    FRAC_CONST(0.1505228305916774),
    FRAC_CONST(0.15203915632824605),
    FRAC_CONST(0.15355512430199345),
    FRAC_CONST(0.15507073094570051),
    FRAC_CONST(0.15658597269299843),
    FRAC_CONST(0.15810084597837698),
    FRAC_CONST(0.15961534723719306),
    FRAC_CONST(0.16112947290567881),
    FRAC_CONST(0.16264321942095031),
    FRAC_CONST(0.16415658322101581),
    FRAC_CONST(0.16566956074478412),
    FRAC_CONST(0.16718214843207294),
    FRAC_CONST(0.16869434272361733),
    FRAC_CONST(0.17020614006107807),
    FRAC_CONST(0.17171753688704997),
    FRAC_CONST(0.17322852964507032),
    FRAC_CONST(0.1747391147796272),
    FRAC_CONST(0.17624928873616788),
    FRAC_CONST(0.17775904796110717),
    FRAC_CONST(0.17926838890183575),
    FRAC_CONST(0.18077730800672859),
    FRAC_CONST(0.1822858017251533),
    FRAC_CONST(0.18379386650747845),
    FRAC_CONST(0.1853014988050819),
    FRAC_CONST(0.18680869507035927),
    FRAC_CONST(0.18831545175673212),
    FRAC_CONST(0.18982176531865641),
    FRAC_CONST(0.1913276322116309),
    FRAC_CONST(0.19283304889220523),
    FRAC_CONST(0.1943380118179886),
    FRAC_CONST(0.19584251744765785),
    FRAC_CONST(0.19734656224096592),
    FRAC_CONST(0.19885014265875009),
    FRAC_CONST(0.20035325516294045),
    FRAC_CONST(0.20185589621656805),
    FRAC_CONST(0.20335806228377332),
    FRAC_CONST(0.20485974982981442),
    FRAC_CONST(0.20636095532107551),
    FRAC_CONST(0.20786167522507507),
    FRAC_CONST(0.20936190601047416),
    FRAC_CONST(0.21086164414708486),
    FRAC_CONST(0.21236088610587842),
    FRAC_CONST(0.21385962835899375),
    FRAC_CONST(0.21535786737974555),
    FRAC_CONST(0.21685559964263262),
    FRAC_CONST(0.21835282162334632),
    FRAC_CONST(0.2198495297987787),
    FRAC_CONST(0.22134572064703081),
    FRAC_CONST(0.22284139064742112),
    FRAC_CONST(0.2243365362804936),
    FRAC_CONST(0.22583115402802617),
    FRAC_CONST(0.22732524037303886),
    FRAC_CONST(0.22881879179980222),
    FRAC_CONST(0.23031180479384544),
    FRAC_CONST(0.23180427584196478),
    FRAC_CONST(0.23329620143223159),
    FRAC_CONST(0.23478757805400097),
    FRAC_CONST(0.23627840219791957),
    FRAC_CONST(0.23776867035593419),
    FRAC_CONST(0.23925837902129998),
    FRAC_CONST(0.24074752468858843),
    FRAC_CONST(0.24223610385369601),
    FRAC_CONST(0.24372411301385216),
    FRAC_CONST(0.24521154866762754),
    FRAC_CONST(0.24669840731494241),
    FRAC_CONST(0.24818468545707478),
    FRAC_CONST(0.24967037959666857),
    FRAC_CONST(0.25115548623774192),
    FRAC_CONST(0.25264000188569552),
    FRAC_CONST(0.25412392304732062),
    FRAC_CONST(0.25560724623080738),
    FRAC_CONST(0.25708996794575312),
    FRAC_CONST(0.25857208470317034),
    FRAC_CONST(0.26005359301549519),
    FRAC_CONST(0.26153448939659552),
    FRAC_CONST(0.263014770361779),
    FRAC_CONST(0.26449443242780163),
    FRAC_CONST(0.26597347211287559),
    FRAC_CONST(0.26745188593667762),
    FRAC_CONST(0.26892967042035726),
    FRAC_CONST(0.27040682208654482),
    FRAC_CONST(0.27188333745935972),
    FRAC_CONST(0.27335921306441868),
    FRAC_CONST(0.27483444542884394),
    FRAC_CONST(0.27630903108127108),
    FRAC_CONST(0.27778296655185769),
    FRAC_CONST(0.27925624837229118),
    FRAC_CONST(0.28072887307579719),
    FRAC_CONST(0.28220083719714756),
    FRAC_CONST(0.28367213727266843),
    FRAC_CONST(0.28514276984024867),
    FRAC_CONST(0.28661273143934779),
    FRAC_CONST(0.28808201861100413),
    FRAC_CONST(0.28955062789784303),
    FRAC_CONST(0.29101855584408509),
    FRAC_CONST(0.29248579899555388),
    FRAC_CONST(0.29395235389968466),
    FRAC_CONST(0.29541821710553201),
    FRAC_CONST(0.29688338516377827),
    FRAC_CONST(0.2983478546267414),
    FRAC_CONST(0.29981162204838335),
    FRAC_CONST(0.30127468398431795),
    FRAC_CONST(0.30273703699181914),
    FRAC_CONST(0.30419867762982911),
    FRAC_CONST(0.30565960245896612),
    FRAC_CONST(0.3071198080415331),
    FRAC_CONST(0.30857929094152509),
    FRAC_CONST(0.31003804772463789),
    FRAC_CONST(0.31149607495827591),
    FRAC_CONST(0.3129533692115602),
    FRAC_CONST(0.31440992705533666),
    FRAC_CONST(0.31586574506218396),
    FRAC_CONST(0.31732081980642174),
    FRAC_CONST(0.31877514786411848),
    FRAC_CONST(0.32022872581309986),
    FRAC_CONST(0.32168155023295658),
    FRAC_CONST(0.32313361770505233),
    FRAC_CONST(0.32458492481253215),
    FRAC_CONST(0.32603546814033024),
    FRAC_CONST(0.327485244275178),
    FRAC_CONST(0.3289342498056122),
    FRAC_CONST(0.33038248132198278),
    FRAC_CONST(0.33182993541646111),
    FRAC_CONST(0.33327660868304793),
    FRAC_CONST(0.33472249771758122),
    FRAC_CONST(0.33616759911774452),
    FRAC_CONST(0.33761190948307462),
    FRAC_CONST(0.33905542541496964),
    FRAC_CONST(0.34049814351669716),
    FRAC_CONST(0.34194006039340219),
    FRAC_CONST(0.34338117265211504),
    FRAC_CONST(0.34482147690175929),
    FRAC_CONST(0.34626096975316001),
    FRAC_CONST(0.34769964781905138),
    FRAC_CONST(0.34913750771408497),
    FRAC_CONST(0.35057454605483751),
    FRAC_CONST(0.35201075945981908),
    FRAC_CONST(0.35344614454948081),
    FRAC_CONST(0.35488069794622279),
    FRAC_CONST(0.35631441627440241),
    FRAC_CONST(0.3577472961603419),
    FRAC_CONST(0.3591793342323365),
    FRAC_CONST(0.36061052712066227),
    FRAC_CONST(0.36204087145758418),
    FRAC_CONST(0.36347036387736376),
    FRAC_CONST(0.36489900101626732),
    FRAC_CONST(0.36632677951257359),
    FRAC_CONST(0.36775369600658198),
    FRAC_CONST(0.36917974714062002),
    FRAC_CONST(0.37060492955905167),
    FRAC_CONST(0.37202923990828501),
    FRAC_CONST(0.3734526748367803),
    FRAC_CONST(0.37487523099505754),
    FRAC_CONST(0.37629690503570479),
    FRAC_CONST(0.37771769361338564),
    FRAC_CONST(0.37913759338484732),
    FRAC_CONST(0.38055660100892852),
    FRAC_CONST(0.38197471314656722),
    FRAC_CONST(0.38339192646080866),
    FRAC_CONST(0.38480823761681288),
    FRAC_CONST(0.38622364328186298),
    FRAC_CONST(0.38763814012537273),
    FRAC_CONST(0.38905172481889438),
    FRAC_CONST(0.39046439403612659),
    FRAC_CONST(0.39187614445292235),
    FRAC_CONST(0.3932869727472964),
    FRAC_CONST(0.39469687559943356),
    FRAC_CONST(0.39610584969169627),
    FRAC_CONST(0.39751389170863233),
    FRAC_CONST(0.39892099833698291),
    FRAC_CONST(0.40032716626569009),
    FRAC_CONST(0.40173239218590501),
    FRAC_CONST(0.4031366727909953),
    FRAC_CONST(0.404540004776553),
    FRAC_CONST(0.40594238484040251),
    FRAC_CONST(0.40734380968260797),
    FRAC_CONST(0.40874427600548136),
    FRAC_CONST(0.41014378051359024),
    FRAC_CONST(0.41154231991376522),
    FRAC_CONST(0.41293989091510808),
    FRAC_CONST(0.4143364902289991),
    FRAC_CONST(0.41573211456910536),
    FRAC_CONST(0.41712676065138787),
    FRAC_CONST(0.4185204251941097),
    FRAC_CONST(0.41991310491784362),
    FRAC_CONST(0.42130479654547964),
    FRAC_CONST(0.42269549680223295),
    FRAC_CONST(0.42408520241565156),
    FRAC_CONST(0.4254739101156238),
    FRAC_CONST(0.42686161663438643),
    FRAC_CONST(0.42824831870653196),
    FRAC_CONST(0.42963401306901638),
    FRAC_CONST(0.43101869646116703),
    FRAC_CONST(0.43240236562469014),
    FRAC_CONST(0.43378501730367852),
    FRAC_CONST(0.43516664824461926),
    FRAC_CONST(0.4365472551964012),
    FRAC_CONST(0.43792683491032286),
    FRAC_CONST(0.43930538414009995),
    FRAC_CONST(0.4406828996418729),
    FRAC_CONST(0.4420593781742147),
    FRAC_CONST(0.44343481649813848),
    FRAC_CONST(0.44480921137710488),
    FRAC_CONST(0.44618255957703007),
    FRAC_CONST(0.44755485786629301),
    FRAC_CONST(0.44892610301574326),
    FRAC_CONST(0.45029629179870861),
    FRAC_CONST(0.45166542099100249),
    FRAC_CONST(0.45303348737093158),
    FRAC_CONST(0.45440048771930358),
    FRAC_CONST(0.45576641881943464),
    FRAC_CONST(0.45713127745715698),
    FRAC_CONST(0.45849506042082627),
    FRAC_CONST(0.45985776450132954),
    FRAC_CONST(0.46121938649209238),
    FRAC_CONST(0.46257992318908681),
    FRAC_CONST(0.46393937139083852),
    FRAC_CONST(0.4652977278984346),
    FRAC_CONST(0.46665498951553092),
    FRAC_CONST(0.46801115304835983),
    FRAC_CONST(0.46936621530573752),
    FRAC_CONST(0.4707201730990716),
    FRAC_CONST(0.47207302324236866),
    FRAC_CONST(0.47342476255224153),
    FRAC_CONST(0.47477538784791712),
    FRAC_CONST(0.47612489595124358),
    FRAC_CONST(0.47747328368669806),
    FRAC_CONST(0.47882054788139389),
    FRAC_CONST(0.48016668536508839),
    FRAC_CONST(0.48151169297018986),
    FRAC_CONST(0.48285556753176567),
    FRAC_CONST(0.48419830588754903),
    FRAC_CONST(0.48553990487794696),
    FRAC_CONST(0.48688036134604734),
    FRAC_CONST(0.48821967213762679),
    FRAC_CONST(0.48955783410115744),
    FRAC_CONST(0.49089484408781509),
    FRAC_CONST(0.49223069895148602),
    FRAC_CONST(0.49356539554877477),
    FRAC_CONST(0.49489893073901126),
    FRAC_CONST(0.49623130138425825),
    FRAC_CONST(0.49756250434931915),
    FRAC_CONST(0.49889253650174459),
    FRAC_CONST(0.50022139471184068),
    FRAC_CONST(0.50154907585267539),
    FRAC_CONST(0.50287557680008699),
    FRAC_CONST(0.50420089443269034),
    FRAC_CONST(0.50552502563188539),
    FRAC_CONST(0.50684796728186321),
    FRAC_CONST(0.5081697162696146),
    FRAC_CONST(0.50949026948493636),
    FRAC_CONST(0.51080962382043904),
    FRAC_CONST(0.51212777617155469),
    FRAC_CONST(0.51344472343654346),
    FRAC_CONST(0.5147604625165012),
    FRAC_CONST(0.51607499031536663),
    FRAC_CONST(0.51738830373992906),
    FRAC_CONST(0.51870039969983495),
    FRAC_CONST(0.52001127510759604),
    FRAC_CONST(0.52132092687859566),
    FRAC_CONST(0.52262935193109661),
    FRAC_CONST(0.5239365471862486),
    FRAC_CONST(0.52524250956809471),
    FRAC_CONST(0.52654723600357944),
    FRAC_CONST(0.52785072342255523),
    FRAC_CONST(0.52915296875779061),
    FRAC_CONST(0.53045396894497632),
    FRAC_CONST(0.53175372092273332),
    FRAC_CONST(0.53305222163261945),
    FRAC_CONST(0.53434946801913752),
    FRAC_CONST(0.53564545702974109),
    FRAC_CONST(0.53694018561484291),
    FRAC_CONST(0.5382336507278217),
    FRAC_CONST(0.53952584932502889),
    FRAC_CONST(0.54081677836579667),
    FRAC_CONST(0.54210643481244392),
    FRAC_CONST(0.5433948156302848),
    FRAC_CONST(0.54468191778763453),
    FRAC_CONST(0.54596773825581757),
    FRAC_CONST(0.54725227400917409),
    FRAC_CONST(0.54853552202506739),
    FRAC_CONST(0.54981747928389091),
    FRAC_CONST(0.55109814276907543),
    FRAC_CONST(0.55237750946709607),
    FRAC_CONST(0.55365557636747931),
    FRAC_CONST(0.55493234046281037),
    FRAC_CONST(0.55620779874873993),
    FRAC_CONST(0.55748194822399155),
    FRAC_CONST(0.55875478589036831),
    FRAC_CONST(0.56002630875276038),
    FRAC_CONST(0.56129651381915147),
    FRAC_CONST(0.56256539810062656),
    FRAC_CONST(0.56383295861137817),
    FRAC_CONST(0.56509919236871398),
    FRAC_CONST(0.56636409639306384),
    FRAC_CONST(0.56762766770798623),
    FRAC_CONST(0.56888990334017586),
    FRAC_CONST(0.5701508003194703),
    FRAC_CONST(0.57141035567885723),
    FRAC_CONST(0.57266856645448116),
    FRAC_CONST(0.57392542968565075),
    FRAC_CONST(0.57518094241484508),
    FRAC_CONST(0.57643510168772183),
    FRAC_CONST(0.5776879045531228),
    FRAC_CONST(0.57893934806308178),
    FRAC_CONST(0.58018942927283168),
    FRAC_CONST(0.58143814524081017),
    FRAC_CONST(0.58268549302866846),
    FRAC_CONST(0.58393146970127618),
    FRAC_CONST(0.58517607232673041),
    FRAC_CONST(0.5864192979763605),
    FRAC_CONST(0.58766114372473666),
    FRAC_CONST(0.58890160664967572),
    FRAC_CONST(0.59014068383224882),
    FRAC_CONST(0.59137837235678758),
    FRAC_CONST(0.59261466931089113),
    FRAC_CONST(0.59384957178543363),
    FRAC_CONST(0.59508307687456996),
    FRAC_CONST(0.59631518167574371),
    FRAC_CONST(0.59754588328969316),
    FRAC_CONST(0.59877517882045872),
    FRAC_CONST(0.60000306537538894),
    FRAC_CONST(0.6012295400651485),
    FRAC_CONST(0.60245460000372375),
    FRAC_CONST(0.60367824230843037),
    FRAC_CONST(0.60490046409991982),
    FRAC_CONST(0.60612126250218612),
    FRAC_CONST(0.60734063464257293),
    FRAC_CONST(0.60855857765177945),
    FRAC_CONST(0.60977508866386843),
    FRAC_CONST(0.61099016481627166),
    FRAC_CONST(0.61220380324979795),
    FRAC_CONST(0.61341600110863859),
    FRAC_CONST(0.61462675554037505),
    FRAC_CONST(0.61583606369598509),
    FRAC_CONST(0.61704392272984976),
    FRAC_CONST(0.61825032979976025),
    FRAC_CONST(0.61945528206692402),
    FRAC_CONST(0.62065877669597214),
    FRAC_CONST(0.62186081085496536),
    FRAC_CONST(0.62306138171540126),
    FRAC_CONST(0.62426048645222065),
    FRAC_CONST(0.62545812224381436),
    FRAC_CONST(0.62665428627202935),
    FRAC_CONST(0.62784897572217646),
    FRAC_CONST(0.629042187783036),
    FRAC_CONST(0.63023391964686437),
    FRAC_CONST(0.63142416850940186),
    FRAC_CONST(0.63261293156987741),
    FRAC_CONST(0.63380020603101728),
    FRAC_CONST(0.63498598909904946),
    FRAC_CONST(0.63617027798371217),
    FRAC_CONST(0.63735306989825913),
    FRAC_CONST(0.63853436205946679),
    FRAC_CONST(0.63971415168764045),
    FRAC_CONST(0.64089243600662138),
    FRAC_CONST(0.64206921224379254),
    FRAC_CONST(0.64324447763008585),
    FRAC_CONST(0.64441822939998838),
    FRAC_CONST(0.64559046479154869),
    FRAC_CONST(0.64676118104638392),
    FRAC_CONST(0.64793037540968534),
    FRAC_CONST(0.64909804513022595),
    FRAC_CONST(0.65026418746036585),
    FRAC_CONST(0.65142879965605982),
    FRAC_CONST(0.65259187897686244),
    FRAC_CONST(0.65375342268593606),
    FRAC_CONST(0.65491342805005603),
    FRAC_CONST(0.6560718923396176),
    FRAC_CONST(0.65722881282864254),
    FRAC_CONST(0.65838418679478505),
    FRAC_CONST(0.65953801151933866),
    FRAC_CONST(0.6606902842872423),
    FRAC_CONST(0.66184100238708687),
    FRAC_CONST(0.66299016311112147),
    FRAC_CONST(0.66413776375526001),
    FRAC_CONST(0.66528380161908718),
    FRAC_CONST(0.66642827400586524),
    FRAC_CONST(0.66757117822254031),
    FRAC_CONST(0.66871251157974798),
    FRAC_CONST(0.66985227139182102),
    FRAC_CONST(0.67099045497679422),
    FRAC_CONST(0.67212705965641173),
    FRAC_CONST(0.67326208275613297),
    FRAC_CONST(0.67439552160513905),
    FRAC_CONST(0.67552737353633852),
    FRAC_CONST(0.67665763588637495),
    FRAC_CONST(0.6777863059956315),
    FRAC_CONST(0.67891338120823841),
    FRAC_CONST(0.68003885887207893),
    FRAC_CONST(0.68116273633879543),
    FRAC_CONST(0.68228501096379557),
    FRAC_CONST(0.68340568010625868),
    FRAC_CONST(0.6845247411291423),
    FRAC_CONST(0.68564219139918747),
    FRAC_CONST(0.68675802828692589),
    FRAC_CONST(0.68787224916668555),
    FRAC_CONST(0.68898485141659704),
    FRAC_CONST(0.69009583241859995),
    FRAC_CONST(0.69120518955844845),
    FRAC_CONST(0.69231292022571822),
    FRAC_CONST(0.69341902181381176),
    FRAC_CONST(0.69452349171996552),
    FRAC_CONST(0.69562632734525487),
    FRAC_CONST(0.6967275260946012),
    FRAC_CONST(0.69782708537677729),
    FRAC_CONST(0.69892500260441415),
    FRAC_CONST(0.70002127519400625),
    FRAC_CONST(0.70111590056591866),
    FRAC_CONST(0.70220887614439187),
    FRAC_CONST(0.70330019935754873),
    FRAC_CONST(0.70438986763740041),
    FRAC_CONST(0.7054778784198521),
    FRAC_CONST(0.70656422914470951),
    FRAC_CONST(0.70764891725568435),
    FRAC_CONST(0.70873194020040065),
    FRAC_CONST(0.70981329543040084),
    FRAC_CONST(0.71089298040115168),
    FRAC_CONST(0.71197099257204999),
    FRAC_CONST(0.71304732940642923),
    FRAC_CONST(0.71412198837156471),
    FRAC_CONST(0.71519496693868001),
    FRAC_CONST(0.71626626258295312),
    FRAC_CONST(0.71733587278352173),
    FRAC_CONST(0.71840379502348972),
    FRAC_CONST(0.71947002678993299),
    FRAC_CONST(0.72053456557390527),
    FRAC_CONST(0.72159740887044366),
    FRAC_CONST(0.72265855417857561),
    FRAC_CONST(0.72371799900132339),
    FRAC_CONST(0.72477574084571128),
    FRAC_CONST(0.72583177722277037),
    FRAC_CONST(0.72688610564754497),
    FRAC_CONST(0.72793872363909862),
    FRAC_CONST(0.72898962872051931),
    FRAC_CONST(0.73003881841892615),
    FRAC_CONST(0.73108629026547423),
    FRAC_CONST(0.73213204179536129),
    FRAC_CONST(0.73317607054783274),
    FRAC_CONST(0.73421837406618817),
    FRAC_CONST(0.73525894989778673),
    FRAC_CONST(0.73629779559405306),
    FRAC_CONST(0.73733490871048279),
    FRAC_CONST(0.73837028680664851),
    FRAC_CONST(0.73940392744620576),
    FRAC_CONST(0.74043582819689802),
    FRAC_CONST(0.74146598663056329),
    FRAC_CONST(0.74249440032313918),
    FRAC_CONST(0.74352106685466912),
    FRAC_CONST(0.74454598380930725),
    FRAC_CONST(0.74556914877532543),
    FRAC_CONST(0.74659055934511731),
    FRAC_CONST(0.74761021311520515),
    FRAC_CONST(0.74862810768624533),
    FRAC_CONST(0.74964424066303348),
    FRAC_CONST(0.75065860965451059),
    FRAC_CONST(0.75167121227376843),
    FRAC_CONST(0.75268204613805523),
    FRAC_CONST(0.75369110886878121),
    FRAC_CONST(0.75469839809152439),
    FRAC_CONST(0.75570391143603588),
    FRAC_CONST(0.75670764653624567),
    FRAC_CONST(0.75770960103026808),
    FRAC_CONST(0.75870977256040739),
    FRAC_CONST(0.75970815877316344),
    FRAC_CONST(0.76070475731923692),
    FRAC_CONST(0.76169956585353527),
    FRAC_CONST(0.76269258203517787),
    FRAC_CONST(0.76368380352750187),
    FRAC_CONST(0.76467322799806714),
    FRAC_CONST(0.76566085311866239),
    FRAC_CONST(0.76664667656531038),
    FRAC_CONST(0.76763069601827327),
    FRAC_CONST(0.76861290916205827),
    FRAC_CONST(0.76959331368542294),
    FRAC_CONST(0.7705719072813807),
    FRAC_CONST(0.7715486876472063),
    FRAC_CONST(0.77252365248444133),
    FRAC_CONST(0.77349679949889905),
    FRAC_CONST(0.77446812640067086),
    FRAC_CONST(0.77543763090413043),
    FRAC_CONST(0.77640531072794039),
    FRAC_CONST(0.7773711635950562),
    FRAC_CONST(0.77833518723273309),
    FRAC_CONST(0.7792973793725303),
    FRAC_CONST(0.78025773775031659),
    FRAC_CONST(0.78121626010627609),
    FRAC_CONST(0.7821729441849129),
    FRAC_CONST(0.78312778773505731),
    FRAC_CONST(0.78408078850986995),
    FRAC_CONST(0.78503194426684808),
    FRAC_CONST(0.78598125276783015),
    FRAC_CONST(0.7869287117790017),
    FRAC_CONST(0.78787431907090011),
    FRAC_CONST(0.78881807241842017),
    FRAC_CONST(0.78975996960081907),
    FRAC_CONST(0.79070000840172161),
    FRAC_CONST(0.79163818660912577),
    FRAC_CONST(0.79257450201540758),
    FRAC_CONST(0.79350895241732666),
    FRAC_CONST(0.79444153561603059),
    FRAC_CONST(0.79537224941706119),
    FRAC_CONST(0.79630109163035911),
    FRAC_CONST(0.7972280600702687),
    FRAC_CONST(0.79815315255554375),
    FRAC_CONST(0.79907636690935235),
    FRAC_CONST(0.79999770095928191),
    FRAC_CONST(0.8009171525373443),
    FRAC_CONST(0.80183471947998131),
    FRAC_CONST(0.80275039962806916),
    FRAC_CONST(0.80366419082692409),
    FRAC_CONST(0.804576090926307),
    FRAC_CONST(0.80548609778042912),
    FRAC_CONST(0.80639420924795624),
    FRAC_CONST(0.80730042319201445),
    FRAC_CONST(0.80820473748019472),
    FRAC_CONST(0.80910714998455813),
    FRAC_CONST(0.81000765858164114),
    FRAC_CONST(0.81090626115245967),
    FRAC_CONST(0.81180295558251536),
    FRAC_CONST(0.81269773976179949),
    FRAC_CONST(0.81359061158479851),
    FRAC_CONST(0.81448156895049861),
    FRAC_CONST(0.81537060976239129),
    FRAC_CONST(0.81625773192847739),
    FRAC_CONST(0.81714293336127297),
    FRAC_CONST(0.81802621197781344),
    FRAC_CONST(0.81890756569965895),
    FRAC_CONST(0.81978699245289899),
    FRAC_CONST(0.82066449016815746),
    FRAC_CONST(0.82154005678059761),
    FRAC_CONST(0.82241369022992639),
    FRAC_CONST(0.82328538846040011),
    FRAC_CONST(0.82415514942082857),
    FRAC_CONST(0.82502297106458022),
    FRAC_CONST(0.82588885134958678),
    FRAC_CONST(0.82675278823834852),
    FRAC_CONST(0.8276147796979384),
    FRAC_CONST(0.82847482370000713),
    FRAC_CONST(0.82933291822078825),
    FRAC_CONST(0.83018906124110237),
    FRAC_CONST(0.83104325074636232),
    FRAC_CONST(0.83189548472657759),
    FRAC_CONST(0.83274576117635946),
    FRAC_CONST(0.83359407809492514),
    FRAC_CONST(0.83444043348610319),
    FRAC_CONST(0.83528482535833737),
    FRAC_CONST(0.83612725172469216),
    FRAC_CONST(0.83696771060285702),
    FRAC_CONST(0.83780620001515094),
    FRAC_CONST(0.8386427179885273),
    FRAC_CONST(0.83947726255457855),
    FRAC_CONST(0.84030983174954077),
    FRAC_CONST(0.84114042361429808),
    FRAC_CONST(0.84196903619438768),
    FRAC_CONST(0.84279566754000412),
    FRAC_CONST(0.84362031570600404),
    FRAC_CONST(0.84444297875191066),
    FRAC_CONST(0.84526365474191822),
    FRAC_CONST(0.84608234174489694),
    FRAC_CONST(0.84689903783439735),
    FRAC_CONST(0.84771374108865427),
    FRAC_CONST(0.84852644959059265),
    FRAC_CONST(0.84933716142783067),
    FRAC_CONST(0.85014587469268521),
    FRAC_CONST(0.85095258748217573),
    FRAC_CONST(0.85175729789802912),
    FRAC_CONST(0.85256000404668397),
    FRAC_CONST(0.85336070403929543),
    FRAC_CONST(0.85415939599173873),
    FRAC_CONST(0.85495607802461482),
    FRAC_CONST(0.85575074826325392),
    FRAC_CONST(0.85654340483771996),
    FRAC_CONST(0.85733404588281559),
    FRAC_CONST(0.85812266953808602),
    FRAC_CONST(0.8589092739478239),
    FRAC_CONST(0.85969385726107261),
    FRAC_CONST(0.86047641763163207),
    FRAC_CONST(0.86125695321806206),
    FRAC_CONST(0.86203546218368721),
    FRAC_CONST(0.86281194269660033),
    FRAC_CONST(0.86358639292966799),
    FRAC_CONST(0.86435881106053403),
    FRAC_CONST(0.86512919527162369),
    FRAC_CONST(0.86589754375014882),
    FRAC_CONST(0.86666385468811102),
    FRAC_CONST(0.86742812628230692),
    FRAC_CONST(0.86819035673433131),
    FRAC_CONST(0.86895054425058238),
    FRAC_CONST(0.86970868704226556),
    FRAC_CONST(0.87046478332539767),
    FRAC_CONST(0.8712188313208109),
    FRAC_CONST(0.8719708292541577),
    FRAC_CONST(0.8727207753559143),
    FRAC_CONST(0.87346866786138488),
    FRAC_CONST(0.8742145050107063),
    FRAC_CONST(0.87495828504885154),
    FRAC_CONST(0.8757000062256346),
    FRAC_CONST(0.87643966679571361),
    FRAC_CONST(0.87717726501859594),
    FRAC_CONST(0.87791279915864173),
    FRAC_CONST(0.87864626748506813),
    FRAC_CONST(0.87937766827195318),
    FRAC_CONST(0.88010699979824036),
    FRAC_CONST(0.88083426034774204),
    FRAC_CONST(0.88155944820914378),
    FRAC_CONST(0.8822825616760086),
    FRAC_CONST(0.88300359904678072),
    FRAC_CONST(0.88372255862478966),
    FRAC_CONST(0.8844394387182537),
    FRAC_CONST(0.88515423764028511),
    FRAC_CONST(0.88586695370889279),
    FRAC_CONST(0.88657758524698704),
    FRAC_CONST(0.88728613058238315),
    FRAC_CONST(0.88799258804780556),
    FRAC_CONST(0.88869695598089171),
    FRAC_CONST(0.88939923272419552),
    FRAC_CONST(0.89009941662519221),
    FRAC_CONST(0.89079750603628149),
    FRAC_CONST(0.89149349931479138),
    FRAC_CONST(0.89218739482298248),
    FRAC_CONST(0.89287919092805168),
    FRAC_CONST(0.89356888600213602),
    FRAC_CONST(0.89425647842231604),
    FRAC_CONST(0.89494196657062075),
    FRAC_CONST(0.89562534883403),
    FRAC_CONST(0.89630662360447966),
    FRAC_CONST(0.89698578927886397),
    FRAC_CONST(0.89766284425904075),
    FRAC_CONST(0.89833778695183419),
    FRAC_CONST(0.89901061576903907),
    FRAC_CONST(0.89968132912742393),
    FRAC_CONST(0.9003499254487356),
    FRAC_CONST(0.90101640315970233),
    FRAC_CONST(0.90168076069203773),
    FRAC_CONST(0.9023429964824442),
    FRAC_CONST(0.90300310897261704),
    FRAC_CONST(0.90366109660924798),
    FRAC_CONST(0.90431695784402832),
    FRAC_CONST(0.90497069113365325),
    FRAC_CONST(0.90562229493982516),
    FRAC_CONST(0.90627176772925766),
    FRAC_CONST(0.90691910797367803),
    FRAC_CONST(0.90756431414983252),
    FRAC_CONST(0.9082073847394887),
    FRAC_CONST(0.90884831822943912),
    FRAC_CONST(0.90948711311150543),
    FRAC_CONST(0.91012376788254157),
    FRAC_CONST(0.91075828104443757),
    FRAC_CONST(0.91139065110412232),
    FRAC_CONST(0.91202087657356823),
    FRAC_CONST(0.9126489559697939),
    FRAC_CONST(0.91327488781486776),
    FRAC_CONST(0.91389867063591168),
    FRAC_CONST(0.91452030296510445),
    FRAC_CONST(0.91513978333968526),
    FRAC_CONST(0.91575711030195672),
    FRAC_CONST(0.91637228239928914),
    FRAC_CONST(0.91698529818412289),
    FRAC_CONST(0.91759615621397295),
    FRAC_CONST(0.9182048550514309),
    FRAC_CONST(0.91881139326416994),
    FRAC_CONST(0.91941576942494696),
    FRAC_CONST(0.92001798211160657),
    FRAC_CONST(0.92061802990708386),
    FRAC_CONST(0.92121591139940873),
    FRAC_CONST(0.92181162518170812),
    FRAC_CONST(0.92240516985220988),
    FRAC_CONST(0.92299654401424625),
    FRAC_CONST(0.92358574627625656),
    FRAC_CONST(0.9241727752517912),
    FRAC_CONST(0.92475762955951391),
    FRAC_CONST(0.9253403078232062),
    FRAC_CONST(0.92592080867176996),
    FRAC_CONST(0.92649913073923051),
    FRAC_CONST(0.9270752726647401),
    FRAC_CONST(0.92764923309258118),
    FRAC_CONST(0.92822101067216944),
    FRAC_CONST(0.92879060405805702),
    FRAC_CONST(0.9293580119099355),
    FRAC_CONST(0.92992323289263956),
    FRAC_CONST(0.93048626567614978),
    FRAC_CONST(0.93104710893559517),
    FRAC_CONST(0.93160576135125783),
    FRAC_CONST(0.93216222160857432),
    FRAC_CONST(0.93271648839814025),
    FRAC_CONST(0.93326856041571205),
    FRAC_CONST(0.93381843636221096),
    FRAC_CONST(0.9343661149437259),
    FRAC_CONST(0.93491159487151609),
    FRAC_CONST(0.93545487486201462),
    FRAC_CONST(0.9359959536368313),
    FRAC_CONST(0.9365348299227555),
    FRAC_CONST(0.93707150245175919),
    FRAC_CONST(0.93760596996099999),
    FRAC_CONST(0.93813823119282436),
    FRAC_CONST(0.93866828489477017),
    FRAC_CONST(0.9391961298195699),
    FRAC_CONST(0.93972176472515334),
    FRAC_CONST(0.94024518837465088),
    FRAC_CONST(0.94076639953639607),
    FRAC_CONST(0.94128539698392866),
    FRAC_CONST(0.94180217949599765),
    FRAC_CONST(0.94231674585656378),
    FRAC_CONST(0.94282909485480271),
    FRAC_CONST(0.94333922528510772),
    FRAC_CONST(0.94384713594709269),
    FRAC_CONST(0.94435282564559475),
    FRAC_CONST(0.94485629319067721),
    FRAC_CONST(0.94535753739763229),
    FRAC_CONST(0.94585655708698391),
    FRAC_CONST(0.94635335108449059),
    FRAC_CONST(0.946847918221148),
    FRAC_CONST(0.94734025733319194),
    FRAC_CONST(0.94783036726210101),
    FRAC_CONST(0.94831824685459909),
    FRAC_CONST(0.94880389496265838),
    FRAC_CONST(0.94928731044350201),
    FRAC_CONST(0.94976849215960668),
    FRAC_CONST(0.95024743897870523),
    FRAC_CONST(0.95072414977378961),
    FRAC_CONST(0.95119862342311323),
    FRAC_CONST(0.95167085881019386),
    FRAC_CONST(0.95214085482381583),
    FRAC_CONST(0.95260861035803324),
    FRAC_CONST(0.9530741243121722),
    FRAC_CONST(0.95353739559083328),
    FRAC_CONST(0.95399842310389449),
    FRAC_CONST(0.95445720576651349),
    FRAC_CONST(0.95491374249913052),
    FRAC_CONST(0.95536803222747024),
    FRAC_CONST(0.95582007388254542),
    FRAC_CONST(0.95626986640065814),
    FRAC_CONST(0.95671740872340305),
    FRAC_CONST(0.9571626997976701),
    FRAC_CONST(0.95760573857564624),
    FRAC_CONST(0.9580465240148186),
    FRAC_CONST(0.9584850550779761),
    FRAC_CONST(0.95892133073321306),
    FRAC_CONST(0.95935534995393079),
    FRAC_CONST(0.9597871117188399),
    FRAC_CONST(0.96021661501196343),
    FRAC_CONST(0.96064385882263847),
    FRAC_CONST(0.96106884214551935),
    FRAC_CONST(0.961491563980579),
    FRAC_CONST(0.9619120233331121),
    FRAC_CONST(0.9623302192137374),
    FRAC_CONST(0.96274615063839941),
    FRAC_CONST(0.96315981662837136),
    FRAC_CONST(0.96357121621025721),
    FRAC_CONST(0.96398034841599411),
    FRAC_CONST(0.96438721228285429),
    FRAC_CONST(0.9647918068534479),
    FRAC_CONST(0.96519413117572472),
    FRAC_CONST(0.96559418430297683),
    FRAC_CONST(0.96599196529384057),
    FRAC_CONST(0.96638747321229879),
    FRAC_CONST(0.96678070712768327),
    FRAC_CONST(0.96717166611467664),
    FRAC_CONST(0.96756034925331436),
    FRAC_CONST(0.9679467556289878),
    FRAC_CONST(0.9683308843324453),
    FRAC_CONST(0.96871273445979478),
    FRAC_CONST(0.9690923051125061),
    FRAC_CONST(0.96946959539741295),
    FRAC_CONST(0.96984460442671483),
    FRAC_CONST(0.97021733131797916),
    FRAC_CONST(0.97058777519414363),
    FRAC_CONST(0.97095593518351797),
    FRAC_CONST(0.97132181041978616),
    FRAC_CONST(0.97168540004200854),
    FRAC_CONST(0.9720467031946235),
    FRAC_CONST(0.97240571902744977),
    FRAC_CONST(0.97276244669568857),
    FRAC_CONST(0.97311688535992513),
    FRAC_CONST(0.97346903418613095),
    FRAC_CONST(0.9738188923456661),
    FRAC_CONST(0.97416645901528032),
    FRAC_CONST(0.97451173337711572),
    FRAC_CONST(0.97485471461870843),
    FRAC_CONST(0.97519540193299037),
    FRAC_CONST(0.97553379451829136),
    FRAC_CONST(0.97586989157834103),
    FRAC_CONST(0.97620369232227056),
    FRAC_CONST(0.97653519596461447),
    FRAC_CONST(0.97686440172531264),
    FRAC_CONST(0.97719130882971228),
    FRAC_CONST(0.97751591650856928),
    FRAC_CONST(0.97783822399805043),
    FRAC_CONST(0.97815823053973505),
    FRAC_CONST(0.97847593538061683),
    FRAC_CONST(0.97879133777310567),
    FRAC_CONST(0.97910443697502925),
    FRAC_CONST(0.97941523224963478),
    FRAC_CONST(0.97972372286559117),
    FRAC_CONST(0.98002990809698998),
    FRAC_CONST(0.98033378722334796),
    FRAC_CONST(0.98063535952960812),
    FRAC_CONST(0.98093462430614164),
    FRAC_CONST(0.98123158084874973),
    FRAC_CONST(0.98152622845866466),
    FRAC_CONST(0.9818185664425525),
    FRAC_CONST(0.98210859411251361),
    FRAC_CONST(0.98239631078608469),
    FRAC_CONST(0.98268171578624086),
    FRAC_CONST(0.98296480844139644),
    FRAC_CONST(0.98324558808540707),
    FRAC_CONST(0.98352405405757126),
    FRAC_CONST(0.98380020570263149),
    FRAC_CONST(0.98407404237077645),
    FRAC_CONST(0.9843455634176419),
    FRAC_CONST(0.9846147682043126),
    FRAC_CONST(0.9848816560973237),
    FRAC_CONST(0.98514622646866223),
    FRAC_CONST(0.98540847869576842),
    FRAC_CONST(0.98566841216153755),
    FRAC_CONST(0.98592602625432113),
    FRAC_CONST(0.98618132036792827),
    FRAC_CONST(0.98643429390162707),
    FRAC_CONST(0.98668494626014669),
    FRAC_CONST(0.98693327685367771),
    FRAC_CONST(0.98717928509787434),
    FRAC_CONST(0.98742297041385541),
    FRAC_CONST(0.98766433222820571),
    FRAC_CONST(0.98790336997297779),
    FRAC_CONST(0.98814008308569257),
    FRAC_CONST(0.98837447100934128),
    FRAC_CONST(0.98860653319238645),
    FRAC_CONST(0.98883626908876354),
    FRAC_CONST(0.98906367815788154),
    FRAC_CONST(0.98928875986462517),
    FRAC_CONST(0.98951151367935519),
    FRAC_CONST(0.98973193907791057),
    FRAC_CONST(0.98995003554160899),
    FRAC_CONST(0.9901658025572484),
    FRAC_CONST(0.99037923961710816),
    FRAC_CONST(0.99059034621895015),
    FRAC_CONST(0.99079912186602037),
    FRAC_CONST(0.99100556606704937),
    FRAC_CONST(0.99120967833625406),
    FRAC_CONST(0.99141145819333854),
    FRAC_CONST(0.99161090516349537),
    FRAC_CONST(0.99180801877740643),
    FRAC_CONST(0.99200279857124452),
    FRAC_CONST(0.99219524408667392),
    FRAC_CONST(0.99238535487085167),
    FRAC_CONST(0.99257313047642881),
    FRAC_CONST(0.99275857046155114),
    FRAC_CONST(0.99294167438986047),
    FRAC_CONST(0.99312244183049558),
    FRAC_CONST(0.99330087235809328),
    FRAC_CONST(0.99347696555278919),
    FRAC_CONST(0.99365072100021912),
    FRAC_CONST(0.99382213829151966),
    FRAC_CONST(0.99399121702332938),
    FRAC_CONST(0.99415795679778973),
    FRAC_CONST(0.99432235722254581),
    FRAC_CONST(0.9944844179107476),
    FRAC_CONST(0.99464413848105071),
    FRAC_CONST(0.99480151855761711),
    FRAC_CONST(0.99495655777011638),
    FRAC_CONST(0.99510925575372611),
    FRAC_CONST(0.99525961214913339),
    FRAC_CONST(0.9954076266025349),
    FRAC_CONST(0.99555329876563847),
    FRAC_CONST(0.99569662829566352),
    FRAC_CONST(0.99583761485534161),
    FRAC_CONST(0.99597625811291779),
    FRAC_CONST(0.99611255774215113),
    FRAC_CONST(0.99624651342231552),
    FRAC_CONST(0.99637812483820021),
    FRAC_CONST(0.99650739168011082),
    FRAC_CONST(0.9966343136438699),
    FRAC_CONST(0.996758890430818),
    FRAC_CONST(0.99688112174781385),
    FRAC_CONST(0.99700100730723529),
    FRAC_CONST(0.99711854682697998),
    FRAC_CONST(0.99723374003046616),
    FRAC_CONST(0.99734658664663323),
    FRAC_CONST(0.99745708640994191),
    FRAC_CONST(0.99756523906037575),
    FRAC_CONST(0.997671044343441),
    FRAC_CONST(0.99777450201016782),
    FRAC_CONST(0.99787561181711015),
    FRAC_CONST(0.99797437352634699),
    FRAC_CONST(0.99807078690548234),
    FRAC_CONST(0.99816485172764624),
    FRAC_CONST(0.99825656777149518),
    FRAC_CONST(0.99834593482121237),
    FRAC_CONST(0.99843295266650844),
    FRAC_CONST(0.99851762110262221),
    FRAC_CONST(0.99859993993032037),
    FRAC_CONST(0.99867990895589909),
    FRAC_CONST(0.99875752799118334),
    FRAC_CONST(0.99883279685352799),
    FRAC_CONST(0.99890571536581829),
    FRAC_CONST(0.99897628335646982),
    FRAC_CONST(0.99904450065942929),
    FRAC_CONST(0.99911036711417489),
    FRAC_CONST(0.99917388256571638),
    FRAC_CONST(0.99923504686459585),
    FRAC_CONST(0.99929385986688779),
    FRAC_CONST(0.99935032143419944),
    FRAC_CONST(0.9994044314336713),
    FRAC_CONST(0.99945618973797734),
    FRAC_CONST(0.99950559622532531),
    FRAC_CONST(0.99955265077945699),
    FRAC_CONST(0.99959735328964838),
    FRAC_CONST(0.9996397036507102),
    FRAC_CONST(0.99967970176298793),
    FRAC_CONST(0.99971734753236219),
    FRAC_CONST(0.99975264087024884),
    FRAC_CONST(0.99978558169359921),
    FRAC_CONST(0.99981616992490041),
    FRAC_CONST(0.99984440549217524),
    FRAC_CONST(0.99987028832898295),
    FRAC_CONST(0.99989381837441849),
    FRAC_CONST(0.99991499557311347),
    FRAC_CONST(0.999933819875236),
    FRAC_CONST(0.99995029123649048),
    FRAC_CONST(0.99996440961811828),
    FRAC_CONST(0.99997617498689761),
    FRAC_CONST(0.9999855873151432),
    FRAC_CONST(0.99999264658070719),
    FRAC_CONST(0.99999735276697821),
    FRAC_CONST(0.99999970586288223)
};
